/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: TKitUser.c $
   Last Modification:
    $Author: MichaelT $
    $Modtime: 6.11.09 15:55 $
    $Revision: 6224 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : yes
 
   Description:
     USER implemented functions called by the cifX Toolkit. This file is a sample for 
     Windows based systems, and is only intended to show how the functions are expected
     to be implemented. This code may change at any time
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
      8        12.04.2011  SS       USER_GetDMAMode() added
      7        01.04.2010  RM       Data types changed for 32/64 Bit usage
      6        27.11.2008  RM       USER_GetOSFile included
      5        15.05.2008  RM       USER_GetBootloaderFile() parameters fixed
      4        24.09.2007  MT       Bugfix:
									                  - parameters in va_start (USER_Trace) were twisted
      3        13.10.2006  RM       Warmstart file definiton and function
                                    USER_GetWarmstartParameters() included
      2        06.10.2006  MT       USER_ErrorTrace now adds a cr/lf to output
      1        07.08.2006  MT       initial version
 
**************************************************************************************/

/*****************************************************************************/
/*! \file TKitUser.c                                                         *
*    USER implemented functions called by the cifX Toolkit                   */
/*****************************************************************************/

#include "cifXToolkit.h"
#include "WarmstartFile.h"
#include "cifXErrors.h"
#include <stdio.h>
#include <assert.h>

/*****************************************************************************/
/* Internally used helper functions                                          */
/*****************************************************************************/

#define CIFX_DRIVER_KEY          "SYSTEM\\CurrentControlSet\\Services\\CIFxDrv"
#define CIFX_DEVICE_FORMAT_STR   "SYSTEM\\CurrentControlSet\\Services\\CIFxDrv\\DeviceConfig\\%u\\%u"
#define CIFX_CHANNEL_FORMAT_STR  "SYSTEM\\CurrentControlSet\\Services\\CIFxDrv\\DeviceConfig\\%u\\%u\\Channel%u"
#define CIFX_FIRMWARE_COUNT_VAL  "ModuleCount"
#define CIFX_FIRMWARE_FORMAT_STR "Module%u"
#define CIFX_CONFIG_COUNT_VAL    "ConfigCount"
#define CIFX_CONFIG_FORMAT_STR   "Config%u"
#define CIFX_INSTALLDIR_VAL      "InstallDir"
#define CIFX_ALIAS_VALUE         "Alias"
#define CIFX_WARMSTART_VALUE     "WarmstartFile"

HKEY OpenDeviceKey(PCIFX_DEVICE_INFORMATION ptDevInfo)
{
  HKEY          hkDevice   = NULL;
  char          szKey[256] = {0};

  sprintf(szKey, 
          CIFX_DEVICE_FORMAT_STR,
          ptDevInfo->ulDeviceNumber,
          ptDevInfo->ulSerialNumber);

  RegOpenKeyEx(HKEY_LOCAL_MACHINE, 
               szKey,
               0,
               KEY_READ,
               &hkDevice);

  return hkDevice;
}

void GetInstallDir(char* szInstalldir, uint32_t ulMaxLen)
{
  HKEY hkDriver = NULL;

  if(ERROR_SUCCESS !=   RegOpenKeyEx(HKEY_LOCAL_MACHINE, 
                        CIFX_DRIVER_KEY,
                        0,
                        KEY_READ,
                        &hkDriver))
  {
    assert(FALSE);

  } else
  {
    DWORD dwType    = REG_NONE;
    DWORD dwDataLen = ulMaxLen;

    if(ERROR_SUCCESS != RegQueryValueEx(hkDriver,
                                        CIFX_INSTALLDIR_VAL,
                                        NULL,
                                        &dwType,
                                        (LPBYTE)szInstalldir, 
                                        &dwDataLen))
    {
      assert(FALSE);
    }

    RegCloseKey(hkDriver);
  }
}

void GetChannelDir(PCIFX_DEVICE_INFORMATION ptDevInfo, char* szFullFileName, uint32_t ulMaxLen)
{
  char szBuffer[256] = {0};
  
  GetInstallDir(szFullFileName, ulMaxLen);
  sprintf(szBuffer,
          "\\%u_%u\\Channel%u\\",
          ptDevInfo->ulDeviceNumber,
          ptDevInfo->ulSerialNumber,
          ptDevInfo->ulChannel);

  strcat(szFullFileName, szBuffer);
}

HKEY OpenChannelKey(PCIFX_DEVICE_INFORMATION ptDevInfo)
{
  HKEY          hkDevice   = NULL;
  char          szKey[256] = {0};

  sprintf(szKey, 
          CIFX_CHANNEL_FORMAT_STR,
          ptDevInfo->ulDeviceNumber,
          ptDevInfo->ulSerialNumber,
          ptDevInfo->ulChannel);

  RegOpenKeyEx(HKEY_LOCAL_MACHINE, 
               szKey,
               0,
               KEY_READ,
               &hkDevice);

  return hkDevice;
}
/*****************************************************************************/



/*****************************************************************************/
/*!  \addtogroup CIFX_TK_USER User specific implementation
*    \{                                                                      */
/*****************************************************************************/

/*****************************************************************************/
/*! Returns the number of firmware files associated with the card/channel,
*   passed as argument.
*   \param ptDevInfo DeviceInfo including channel number, for which the
*                    firmware file count should be read
*   \return number for firmware files, to download. The returned value will
*           be used as maximum value for ulIdx in calls to
*           USER_GetFirmwareFile                                             */
/*****************************************************************************/
uint32_t USER_GetFirmwareFileCount(PCIFX_DEVICE_INFORMATION ptDevInfo)
{
  uint32_t ulRet      = 0;
  /* Check if the card is configured by "Device Number"/"Serial Number" or if the "Slot Number" is used */
  if(0 == ptDevInfo->ptDeviceInstance->ulSlotNumber) 
  {
      /* The card is referenced by its "Device Number" / "Serial Number" */ 
    HKEY          hkChannel  = OpenChannelKey(ptDevInfo);

    if(NULL != hkChannel)
    {

      DWORD dwType    = REG_NONE;
      DWORD dwData    = 0;
      DWORD dwDataLen = sizeof(dwData);

      if( (ERROR_SUCCESS == RegQueryValueEx(hkChannel,
                                            CIFX_FIRMWARE_COUNT_VAL,
                                            NULL,
                                            &dwType,
                                            (LPBYTE)&dwData,
                                            &dwDataLen)) &&
          (dwType == REG_DWORD) )
      {
        ulRet = dwData;
      }

      RegCloseKey(hkChannel);
    }
  } else
  {
    /* The card is referenced by its "Slot number" (see rotary switch on the device) */
    
    /* Define where to find the files in this case !!!!*/
    USER_Trace(NULL, 
                TRACE_LEVEL_ERROR,
                "USER_GetFirmwareFileCount(), Slot number handling not implemented\r\n");
    
    ulRet = 0;    /* No files found */
  }

  return ulRet;
}

/*****************************************************************************/
/*! Returns firmware file information for the given device/channel and Idx
*   passed as argument.
*   \param ptDevInfo  DeviceInfo including channel number, for which the
*                     firmware file should be delivered
*   \param ulIdx      Index of the returned file
*                     (0..USER_GetFirmwareFileCount() - 1)
*   \param ptFileInfo Short and full file name of the firmware. Used in
*                     calls to OS_OpenFile()
*   \return !=0 on success                                                   */
/*****************************************************************************/
int32_t USER_GetFirmwareFile(PCIFX_DEVICE_INFORMATION ptDevInfo, uint32_t ulIdx, PCIFX_FILE_INFORMATION ptFileInfo)
{
  int32_t iRet = 0;

  /* Check if the card is configured by "Device Number"/"Serial Number" or if the "Slot Number" is used */
  if(0 == ptDevInfo->ptDeviceInstance->ulSlotNumber) 
  {
    /* The card is referenced by its "Device Number" / "Serial Number" */ 
    HKEY hkChannel = OpenChannelKey(ptDevInfo);
    
    if(NULL != hkChannel)
    {
      int8_t szBuffer[256] = {0};
      DWORD  dwDataLen = sizeof(ptFileInfo->szShortFileName);

      sprintf(szBuffer, 
              CIFX_FIRMWARE_FORMAT_STR,
              ulIdx);


      if(ERROR_SUCCESS == RegQueryValueEx(hkChannel,
                                          szBuffer,
                                          NULL, 
                                          NULL,
                                          (LPBYTE)ptFileInfo->szShortFileName,
                                          &dwDataLen))
      {
        GetChannelDir(ptDevInfo, ptFileInfo->szFullFileName, sizeof(ptFileInfo->szFullFileName));
        strcat(ptFileInfo->szFullFileName, ptFileInfo->szShortFileName);

        iRet = 1;
      }

      RegCloseKey(hkChannel);
    }
  } else
  {
    /* The card is referenced by its "Slot number" (see rotary switch on the device) */
    
    /* Define where to find the files in this case !!!!*/
    USER_Trace(NULL, 
                TRACE_LEVEL_ERROR,
                "USER_GetFirmwareFile(), Slot number handling not implemented\r\n");
    
    iRet = 0;    /* No files found */
  }

  return iRet;
}

/*****************************************************************************/
/*! Returns the number of configuration files associated with the card/
*   channel, passed as argument.
*   \param ptDevInfo DeviceInfo including channel number, for which the
*                    configuration file count should be read
*   \return number for confgiuration files, to download. The returned value
*           will be used as maximum value for ulIdx in calls to
*           USER_GetConfgirationFile                                         */
/*****************************************************************************/
uint32_t USER_GetConfigurationFileCount(PCIFX_DEVICE_INFORMATION ptDevInfo)
{
  uint32_t ulRet      = 0;

  /* Check if the card is configured by "Device Number"/"Serial Number" or if the "Slot Number" is used */
  if(0 == ptDevInfo->ptDeviceInstance->ulSlotNumber) 
  {
    /* The card is referenced by its "Device Number" / "Serial Number" */ 
    HKEY          hkChannel  = OpenChannelKey(ptDevInfo);

    if(NULL != hkChannel)
    {
      DWORD dwType    = REG_NONE;
      DWORD dwData    = 0;
      DWORD dwDataLen = sizeof(dwData);

      if( (ERROR_SUCCESS == RegQueryValueEx(hkChannel,
                                            CIFX_CONFIG_COUNT_VAL,
                                            NULL,
                                            &dwType,
                                            (LPBYTE)&dwData,
                                            &dwDataLen)) &&
          (dwType == REG_DWORD) )
      {
        ulRet = dwData;
      }

      RegCloseKey(hkChannel);
    }
  } else
  {
    /* The card is referenced by its "Slot number" (see rotary switch on the device) */
    
    /* Define where to find the files in this case !!!!*/
    USER_Trace(NULL, 
                TRACE_LEVEL_ERROR,
                "USER_GetConfigurationFileCount(), Slot number handling not implemented\r\n");
    
    ulRet = 0;    /* No files found */
  }

  return ulRet;
}

/*****************************************************************************/
/*! Returns configuration file information for the given device/channel and
*   Idx passed as argument.
*   \param ptDevInfo  DeviceInfo including channel number, for which the
*                     configuration file should be delivered
*   \param ulIdx      Index of the returned file
*                     (0..USER_GetConfigurationFileCount() - 1)
*   \param ptFileInfo Short and full file name of the configuration. Used in
*                     calls to OS_OpenFile()
*   \return !=0 on success                                                   */
/*****************************************************************************/
int32_t USER_GetConfigurationFile(PCIFX_DEVICE_INFORMATION ptDevInfo, uint32_t ulIdx, PCIFX_FILE_INFORMATION ptFileInfo)
{
  int32_t iRet = 0;

  /* Check if the card is configured by "Device Number"/"Serial Number" or if the "Slot Number" is used */
  if(0 == ptDevInfo->ptDeviceInstance->ulSlotNumber) 
  {
    /* The card is referenced by its "Device Number" / "Serial Number" */     
    HKEY hkChannel = OpenChannelKey(ptDevInfo);

    if(NULL != hkChannel)
    {
      char szBuffer[256] = {0};
      DWORD dwDataLen = sizeof(ptFileInfo->szShortFileName);

      sprintf(szBuffer, 
              CIFX_CONFIG_FORMAT_STR,
              ulIdx);


      if(ERROR_SUCCESS == RegQueryValueEx(hkChannel,
                                          szBuffer,
                                          NULL, 
                                          NULL,
                                          (LPBYTE)ptFileInfo->szShortFileName,
                                          &dwDataLen))
      {

        GetChannelDir(ptDevInfo, ptFileInfo->szFullFileName, sizeof(ptFileInfo->szFullFileName));
        strcat(ptFileInfo->szFullFileName, ptFileInfo->szShortFileName);

        iRet = 1;
      }

      RegCloseKey(hkChannel);
    }
  } else
  {
    /* The card is referenced by its "Slot number" (see rotary switch on the device) */
    
    /* Define where to find the files in this case !!!!*/
    USER_Trace(NULL, 
                TRACE_LEVEL_ERROR,
                "USER_GetConfigurationFile(), Slot number handling not implemented\r\n");
                
    iRet = 0;    /* No files found */
  }

  return iRet;
}

/*****************************************************************************/
/*! Returns OS file information for the given device/channel and Idx
*   passed as argument. This is needed for module downloading
*   \param ptDevInfo  DeviceInfo for which the
*                     firmware file should be delivered
*   \param ptFileInfo Short and full file name of the firmware. Used in
*                     calls to OS_OpenFile()
*   \return != 0 on success                                                   */
/*****************************************************************************/
int32_t USER_GetOSFile(PCIFX_DEVICE_INFORMATION ptDevInfo, PCIFX_FILE_INFORMATION ptFileInfo)
{
  UNREFERENCED_PARAMETER(ptDevInfo);
  UNREFERENCED_PARAMETER(ptFileInfo);
  /*NOTE: This is currently only possible on PCI cards. As Handling PCI cards in user mode
          does not work, this function has been left blank. */
  return 0; 
}

/*****************************************************************************/
/*! Retrieve the full file name of the cifX Romloader binary image
*   \param ptDevInstance  Pointer to the device instance
*   \param ptFileInfo Short and full file name of the bootloader. Used in
*                     calls to OS_OpenFile()
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
void USER_GetBootloaderFile(PDEVICEINSTANCE ptDevInstance, PCIFX_FILE_INFORMATION ptFileInfo)
{
  char szBuffer[256] = {0};

  GetInstallDir( szBuffer, sizeof(szBuffer));

  /* Check Chip type */
  switch (ptDevInstance->eChipType)
  {
    case eCHIP_TYPE_UNKNOWN:
      OS_Strncpy(ptFileInfo->szShortFileName, "NETX100-BSL.bin", sizeof(ptFileInfo->szShortFileName));
    break;

    case eCHIP_TYPE_NETX500:
    case eCHIP_TYPE_NETX100:
       OS_Strncpy(ptFileInfo->szShortFileName, "NETX100-BSL.bin", sizeof(ptFileInfo->szShortFileName));
    break;

    case eCHIP_TYPE_NETX50:
      OS_Strncpy(ptFileInfo->szShortFileName, "NETX50-BSL.bin", sizeof(ptFileInfo->szShortFileName));
    break;

    case eCHIP_TYPE_NETX51:
      OS_Strncpy(ptFileInfo->szShortFileName, "NETX51-BSL.bin", sizeof(ptFileInfo->szShortFileName));
    break;
  }  
  
  sprintf(ptFileInfo->szFullFileName, "%s\\%s", szBuffer, ptFileInfo->szShortFileName);
}

/*****************************************************************************/
/*! Retrieve the alias name of a cifX Board depending on the Device and
*   Serialnumber passed during this call
*   \param ptDevInfo Device- and Serial number of the card
*   \param ulMaxLen  Maximum length of alias
*   \param szAlias   Buffer to copy alias to. A string of length 0 means
*                    no alias                                                */
/*****************************************************************************/
void USER_GetAliasName(PCIFX_DEVICE_INFORMATION ptDevInfo, uint32_t ulMaxLen, char* szAlias)
{
  HKEY hkDevice = OpenDeviceKey(ptDevInfo);

  OS_Memset(szAlias, 0, ulMaxLen);

  if(NULL != hkDevice)
  {
    char  szBuffer[32] = {0};
    DWORD dwType       = REG_NONE;
    DWORD dwDataLen    = sizeof(szBuffer);

    if( (ERROR_SUCCESS == RegQueryValueEx(hkDevice,
                                          CIFX_ALIAS_VALUE,
                                          NULL,
                                          &dwType,
                                          (LPBYTE)szBuffer,
                                          &dwDataLen)) &&
        (dwType == REG_SZ) )
    {
      OS_Strncpy(szAlias, szBuffer, ulMaxLen);
    }


    RegCloseKey(hkDevice);
  }
}

/*****************************************************************************/
/*! Read the warmstart data from a given warmstart file
*   \param ptDevInfo Device- and Serial number of the card
*   \param ptPacket  Buffer for the warmstart packet                         */
/*****************************************************************************/
int32_t USER_GetWarmstartParameters(PCIFX_DEVICE_INFORMATION ptDevInfo, CIFX_PACKET* ptPacket)
{
  int32_t fRet = 0;
  HKEY hkChannel = OpenChannelKey(ptDevInfo);

  if(NULL != hkChannel)
  {
    int8_t  szBuffer[MAX_PATH] = {0};
    DWORD   dwType             = REG_NONE;
    DWORD   dwDataLen          = sizeof(szBuffer);

    if( (ERROR_SUCCESS == RegQueryValueEx(hkChannel,
                                          CIFX_WARMSTART_VALUE,
                                          NULL,
                                          &dwType,
                                          (LPBYTE)szBuffer,
                                          &dwDataLen)) &&
        (dwType == REG_SZ) )
    {
      int8_t   szFileName[MAX_PATH] = {0};
      uint32_t ulFileLength         = 0;
      void*    pvFile               = NULL;
      
      GetChannelDir( ptDevInfo, szFileName, sizeof(szFileName)); 
      strcat(szFileName, szBuffer);
      
      pvFile = OS_FileOpen( szFileName, &ulFileLength);
      if ( (pvFile == NULL)                   ||
           (ulFileLength < sizeof(CIFX_WS_FILEHEADER)) )
      {
        USER_Trace(NULL, 
                   TRACE_LEVEL_ERROR,
                   "Error opening warmstart file! (lRet = 0x%08X)\r\n", 
                   CIFX_FILE_OPEN_FAILED);
      } else 
      {
        CIFX_WS_FILEHEADER tHeader = {0}; 
        /* Read file header */
        if (OS_FileRead( pvFile, 0, sizeof(tHeader), &tHeader) != sizeof(tHeader))
        {
          USER_Trace(NULL, 
                    TRACE_LEVEL_ERROR,
                    "Error reading from warmstart file! (lRet = 0x%08X)\r\n", 
                    CIFX_FILE_READ_ERROR);

        } else if( tHeader.ulCookie != CIFX_WS_WARMSTART_FILE_COOKIE)
        {
          USER_Trace(NULL, 
                    TRACE_LEVEL_ERROR,
                    "Invalid warmstart file cookie!\r\n");

        } else if( tHeader.ulDataLen > sizeof(*ptPacket))
        {
          USER_Trace(NULL, 
                    TRACE_LEVEL_ERROR,
                    "Invalid warmstart file length!\r\n");
        } else
        {
          /* Read file data */
          if (OS_FileRead( pvFile, sizeof(tHeader), tHeader.ulDataLen, (void*)ptPacket) != tHeader.ulDataLen)
          {
            USER_Trace(NULL, 
                      TRACE_LEVEL_ERROR,
                      "Error reading user data from warmstart file!\r\n");
          } else
          { 
            fRet = 1;
          }  
        }
      }
      
      if( NULL != pvFile) 
        OS_FileClose(pvFile);
    }

    RegCloseKey(hkChannel);
  }
  
  return fRet;
}

/*****************************************************************************/
/*! User trace function
*   right while cifXTKitAddDevice is being processed
*   \param ptDevInstance  Device instance
*   \param ulTraceLevel   Trace level
*   \param szFormat       Format string                                      */
/*****************************************************************************/
void USER_Trace(PDEVICEINSTANCE ptDevInstance, uint32_t ulTraceLevel, const char* szFormat, ...)
{
  va_list vaList;

  va_start(vaList, szFormat);
  vprintf(szFormat, vaList);

  UNREFERENCED_PARAMETER(ulTraceLevel);
  UNREFERENCED_PARAMETER(ptDevInstance);
}

/*****************************************************************************/
/*! Check if interrupts should be enabled for this device
*   \param ptDevInfo  Device Information
*   \return !=0 to enable interrupts                                         */
/*****************************************************************************/
int32_t USER_GetInterruptEnable(PCIFX_DEVICE_INFORMATION ptDevInfo)
{
  /* we cannot handle interrupts in win32 user mode */
  UNREFERENCED_PARAMETER(ptDevInfo);

  return 0;
}

#ifdef CIFX_TOOLKIT_DMA
/*****************************************************************************/
/*! Check if dma should be enabled for this device
*   \param ptDevInfo  Device Information
*   \return CIFX_DMA_STATE_ON/CIFX_DMA_STATE_OFF                             */
/*****************************************************************************/
int32_t USER_GetDMAMode(PCIFX_DEVICE_INFORMATION ptDevInfo)
{
  /* we cannot handle dma in win32 user mode */
  UNREFERENCED_PARAMETER(ptDevInfo);

  return CIFX_DMA_STATE_OFF;
}
#endif

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
